''******************************
''*  Simple Async Driver       *
''*  (c) 2006 Parallax, Inc.   * 
''* Simplified by Matteo Borri *
''******************************                                                                
''
''
'' Bit-bang serial driver for low baud rate (~19.2K) devices - Trimmed, we just need RX
''
'' Authors... Chip Gracey, Phil Pilgrim, Jon Williams, Matteo Borri
''
'' This driver is designed to be method-compatible with the FullDuplex serial object, allowing
'' it to be used when high speed comms or devoting an independent cog for serial I/O is not
'' necessary. 
''
'' To specify inverted baud (start bit = 1), use a negative baud value:
''
''     serial.start(0, 1, -9600)
''
'' If bi-directional communication is desired on the same pin, the serial line should be pulled-up
'' for true mode, or pulled-down for inverted mode as the tx pin will be placed into a hi-z (input)
'' state at the end of transmition to prevent an electrical conflict.
''
'' If only one side is required (e.g., just serial output), use -1 for the unused pin:
''
''     serial.start(-1, 0, 19_200)
''
'' Tested to 19.2 kbaud with clkfreq of 80 MHz (5 MHz crystal, 16x PLL)
'' consider putting this stuff in each definition....


VAR

  long  sin, inverted, started
  long bitTime, b, t, timer


PUB start(rxPin, txpin, baud) ' txpin kept for compatibility with simple_serial

  stop                                                  ' clean-up if restart
  sin := -1

  started~
  
  if lookdown(rxPin : 0..31)                            ' qualify rx pin
    sin := rxPin                                        ' save it
    started := true                           ' set flags

  if started
    dira[sin]~
    inverted := (baud < 0)                              ' set inverted flag
    bitTime := clkfreq / ||baud                         ' calculate serial bit time  
  
  return started


PUB stop
  started~
  sin~
  bitTime~
  inverted~

PUB rx
  return rxtime(-1)

{
  t~
  b~
  timer~

  if started
      waitpeq(inverted & |< sin, |< sin, 0)
    t := cnt + bitTime >> 1                             ' sync + 1/2 bit
    repeat 8
      waitcnt(t += bitTime)                             ' wait for middle of bit
      b := ina[sin] << 7 | b >> 1                       ' sample bit 
    waitcnt(t + bitTime)                                ' allow for stop bit 

    return (b ^ inverted) & $FF                         ' adjust for mode and strip off high bits
}

PUB rxtime(timeout)

  t~
  b~
  timer~

'' Receive a byte; blocks program until byte received
  if started
'    dira[sin]~                                          ' make rx pin an input

    if (timeout == -1)
      waitpeq(inverted & |< sin, |< sin, 0)
{
      if (inverted)
        repeat
        while ina[sin] == 0
      else
        repeat
        while ina[sin] == 1
}

    else     
      if (inverted)
        repeat
          if (++timer == timeout)
             return -1
        while ina[sin] == 0
      else
        repeat
          if (++timer == timeout)
             return -1
        while ina[sin] == 1
    t := cnt + bitTime >> 1                             ' sync + 1/2 bit
    repeat 8
      waitcnt(t += bitTime)                             ' wait for middle of bit
      b := ina[sin] << 7 | b >> 1                       ' sample bit 
    waitcnt(t + bitTime)                                ' allow for stop bit 

    return (b ^ inverted) & $FF                         ' adjust for mode and strip off high bits